/* --------------------------------------------------------------
  Colorpicker.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

/**
 * Colorpicker Widget
 *
 * The user can select a value from the colorpicker or enter his own string which must not be validated
 * because it can also be a SASS mixin or something similar.
 */
export default class Colorpicker {
	/**
	 * Class Constructor
	 *
	 * @param {Setting} entity
	 * @param {StyleConfiguration} styleConfiguration
	 */
	constructor(entity, styleConfiguration) {
		StyleEdit.Validator.isObject(entity);
		StyleEdit.Validator.isObject(styleConfiguration);
		
		/**
		 * @type {Setting}
		 */
		this.entity = entity;
		
		/**
		 * @type {StyleConfiguration}
		 */
		this.styleConfiguration = styleConfiguration;
	}
	
	/**
	 * Initialize the widget.
	 */
	initialize() {
		const color = this.entity.getValue();
		const $element = $('.edit-options-container').find('#' + this.entity.getName());
		const $preview = $element.parents('.form-group').find('.color-preview');
		
		$preview.spectrum({
			color: this.entity.getValue(),
			preferredFormat: 'hex',
			clickoutFiresChange: true,
			showInitial: true,
			showAlpha: true,
			showPalette: true,
			showInput: true,
			chooseText: 'OK',
			cancelText: StyleEdit.Language.translate('option_cancel', 'style_edit'),
			palette: [],
			maxSelectionSize: 20,
			containerClassName: 'colorpicker-widget',
			localStorageKey: StyleEdit.Config.get('localStorageColorPalette'),
			change: color => {
				$element.val(color).trigger('change');
				this._updatePreviewColor($preview, color.toString());
			}
		});
		
		this._updatePreviewColor($preview, color);
		
		$element.change(() => {
			let found = false;
			
			// Update the style configuration.
			$.each(this.styleConfiguration.getSettings(), (index, setting) => {
				$.each(setting.getEntries(), (index, entry) => {
					if (entry.getName() === this.entity.getName()) {
						entry.setValue($element.val());
						found = true;
						return false;
					}
				});
				
				if (found) {
					return false;
				}
			});
			
			this._updatePreviewColor($preview, $element.val());
		});
		
		// Add setting tags to the parent ".form-group" element.
		if (this.entity.getTags() !== null) {
			const tags = this.entity.getTags().join('|');
			$element.parents('.form-group').attr('data-tags', tags);
		}
	}
	
	/**
	 * Update the background color of the selector.
	 *
	 * This method will set the new color or display the no-preview icon if the color
	 * cannot be directly rendered (e.g. is a SASS command).
	 *
	 * @param {jQuery} $preview
	 * @param {String} color
	 *
	 * @private
	 */
	_updatePreviewColor($preview, color) {
		StyleEdit.Validator.isObject($preview);
		StyleEdit.Validator.isString(color);
		
		if (color.match(/(#|rgb|rgba)/ig)) {
			$preview.html('');
			$preview.css('background-color', color);
		} else {
			$preview.html('<i class="fa fa-ban"></i>');
			$preview.css('background-color', '#FFF'); // Hide the text from behind the color-preview element.
		}
	}
	
	/**
	 * Get Widget HTML
	 *
	 * @return {String}
	 */
	getHtml() {
		const template = $('#widget-colorpicker-template').html();
		return Mustache.render(template, this.entity);
	}
}
